% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program Quest.m

function Quest

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenDistance = 30; 	% in inches
p.ScreenHeight = 15; 	% in inches
p.ScreenGamma = 2;	% from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5;

% Open the display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
%Prepare setup of imaging pipeline for onscreen window. 
PsychImaging('PrepareConfiguration'); % First step in starting pipeline
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit floatingpoint framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange'); % normalize the color range ([0, 1] corresponds to [min, max])
PsychImaging('AddTask', 'General', 'EnablePseudoGrayOutput'); % enable high gray level resolution output with bitstealing
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');  % setup Gamma correction method using simple power function for all color channels 
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % Finishes the setup phase for imaging pipeline, creates an onscreen window, performs all remaining configuration steps
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);  % set Gamma for all color channels
HideCursor;  % Hide the mouse cursor

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, 24);

%% Experimental Module

% Specify the stimulus
p.stimSize = 4;         % image size in visual degrees
p.sf = 2; % cycles/degree
p.stimDuration = 0.2;   % stimulus duration in seconds
p.fixDuration = 0.2;    % seconds of fixation prior to stim
p.ITI = 0.5;
p.pThreshold = 0.75;    % performance level to measure the threshold
nTrials = 80;
keys = {'left' 'right' 'esc'}; % keys to respond for left 
                               % and right, and to break

% Compute stimulus parameters
ppd = pi/180 * p.ScreenDistance / p.ScreenHeight * ...
      p.ScreenRect(4);  % pixels per degree
m = round(p.stimSize * ppd);   % image size in pixels
sf = p.sf / ppd;
sigma = m / 6;
fixLen = 1 / 4;         % cross length relative to half of
                        % image size
fixXY(1, :) = [-(1+fixLen) -1 1 1+fixLen 0 0 0 0]*m/2 + ...
      p.ScreenRect(3) / 2;
fixXY(2, :) = [0 0 0 0 -(1+fixLen) -1 1 1+fixLen]*m/2 + ...
      p.ScreenRect(4) / 2;
p.randSeed = ClockRandSeed;     % use clock to set seed for
                        % the random number generator

% procedural gabor allows us to change its parameters 
% very quickly 
tex = CreateProceduralGabor(windowPtr, m, m, 0, ...
      [1 1 1 0] * 0.5, 1, 0.5);
     
% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'tiltLeft' 'logContrast' ...
      'respCorrect' 'respTime' 'Questmean' 'QuestSd'};
rec = nan(nTrials, length(p.recLabel)); 
        % matrix rec initialized with NaN
rec(:, 1) = 1 : nTrials; 
        % count the trial numbers from 1 to nTrials
tiltLeft = ones(nTrials, 1);
tiltLeft(1 : round(nTrials / 2)) = 0;
rec(:, 2) = Shuffle(tiltLeft); 
        % shuffle left and right orientations

% Initialize Quest. Parameters: thresholdGuess priorSd
% pCorrect eta delta gamma
q = QuestCreate(log10(0.5), 3.5, p.pThreshold, 3.5, 0.01, ...
      0.5);

% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

% Start experiment with instructions
str = ['Press left and right arrow keys for top tilt ' ...
       'directions.\n\n' 'Press SPACE to start.'];
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
% Draw Instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
WaitTill('space');      % wait for SPACE to start
Secs = Screen('Flip', windowPtr);
p.start = datestr(now); % record start time

% Run nTrials trials
for i = 1 : nTrials
    % parameters for this trial
    tiltLeft = rec(i, 2);
    ang = (tiltLeft - 0.5) * 90; % tilt angle from vertical
    con = 10 ^ QuestQuantile(q); % get contrast
    con = min(0.5, con);
    WaitTill(Secs + p.ITI);
    
    Screen('DrawLines', windowPtr, fixXY, 3, 0);
    t0 = Screen('Flip', windowPtr, 0, 1); % show cross
    
    Screen('DrawTexture', windowPtr, tex, [], [], ang, 1, ...
           1, [], [], 2, [180 sf sigma con 1 0 0 0]);
    t0 = Screen('Flip', windowPtr, t0 + p.fixDuration); 
        % turn on stimulus
    Screen('Flip', windowPtr, t0 + p.stimDuration); 
        % turn off stimulus
    
    [key Secs] = WaitTill(keys);  % wait till response
    if iscellstr(key), key = key{1}; end 
        % take the first in case of multiple keys
    if strcmp(key, 'esc'), break; end % to stop
    correct = strcmp(key, keys{1}) == tiltLeft; 

    % Update Quest and store data
    q = QuestUpdate(q, log10(con), correct);
    rec(i, 3:7) = [log10(con) correct Secs-t0 ...
        QuestMean(q) QuestSd(q)];
end

p.finish = datestr(now);   % record finish time
save Quest_rst.mat rec p;  % save the results


%% System Reinstatement Module

Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table

